function [V,D,val] = scfiter(EvalG,EvalH,p,tol,maxit,V0,shift,opt)
%Tries to find solutions for G(V)*V=H(V)*V*D.
%EvalG: Function that for a V matrix returns G(V)
%EvalH: Function that returns H(V) for a V matrix
%p: Column number of V
%tol: precision for termination condition
%maxit: maximum number of iterations
%V0: starting matrix
%shift: boolean, whether a shift of the EW of (G,H) should be performed
%opt: 'smallest'/'largest', whether the smallest (min) or the
%     largest (max) eigenvalues are taken.

V = V0;
D = eye(p);
n = size(V,1);
val = zeros(1,maxit); %Saves the objective function value over the iterations

for k = 1:maxit

    G = EvalG(V);
    H = EvalH(V);

    val(k) = trace(V'*G*V)/trace(V'*H*V);

    if norm(G*V-H*V*D)/n < tol %Termination criterion
        break
    end

    %In order to expect convergence at all we need (V,D) so that
    %V is again EW to the smallest/roughest EW D of (G(V),H(V)). Shift ensures
    %that if V occur as EV, they also belong to the p smallest/largest EW
    %belong.

    if strcmp(opt,'smallest')
        if shift
            [~,L] = eigs(G,H);
            sk = 1.01 * max(diag(L)) - min(diag(L));
            for i = 1:p
                G = G - sk/(V(:,i)'*H*V(:,i)) * H*V(:,i)*V(:,i)'*H;
            end
        end
       [V,D] = eig(G,H);
       [~,ind] = sort(diag(D));
       D = D(ind,ind); %sort eigenvalues by size
       V = V(:,ind); %sort eigenvectors accordingly
       V = V(:,1:p); %select 'smallest' eigenvectors
       D = D(1:p,1:p);

    elseif strcmp(opt,'largest')
        if shift
            [~,L] = eig(G,H);
            sk =  - 1.01 * max(diag(L)) + min(diag(L));
            for i = 1:p
                G = G - sk/(V(:,i)'*H*V(:,i)) * H*V(:,i)*V(:,i)'*H;
            end
        end
       [V,D] = eig(G,H);
       [~,ind] = sort(diag(D));
       D = D(ind,ind); %sort eigenvalues by size
       V = V(:,ind); %sort eigenvectors accordingly
       V = V(:,n-p+1:end); %select 'smallest' eigenvectors
       D = D(n-p+1:end,n-p+1:end);
    end


    %H orthonomy columns from V
    DD = diag(V'*H*V);
    DD = 1./sqrt(DD);
    DD = diag(DD);
    V = V*DD;
end
val = val(1:k);
val = [val, trace(V'*EvalG(V)*V)/trace(V'*EvalH(V)*V)];
