function [V,val] = newscf(EvalG,EvalH,p,tol,maxit1,maxit2,V0,shift)
%Calculates the minimum of tr(V'G(V)V)/tr(V'H(V)V) using a
%Newton method
%EvalG: Function that returns G(V) for a V
%EvalH: Function that returns H(V) for a V
%p: column number of V
%tol: Accuracy for termination condition
%maxit1: maximum number of iterations of the Newton method
%maxit2: maximum number of iterations of the called SCF procedure
% ATTENTION: For maxit=0 the number of iterations decreases slowly from 128 onwards.
% ATTENTION: For maxit=-1 the number of iterations increases slowly from 8 onwards
%V0: start matrix
%shift: boolean, whether a shift should be used in the SCF procedure

V = V0;
n = size(V0,1);
val = zeros(1,maxit1);

rho = trace(V'*EvalG(V)*V)/trace(V'*EvalH(V)*V);

for k = 1:maxit1

    G = EvalG(V);
    H = EvalH(V);

    rho = rho + trace(V'*(G-rho*H)*V)/trace(V'*H*V); %Newtonstep
    val(k) = rho; %save current objective function value

    D = @(V) EvalG(V) - rho*EvalH(V);
    I = @(V) eye(n);

    if maxit2 > 0
        [V,~,~] = scfiter(D,I,p,tol,maxit2,V,shift,'smallest'); %determine V such that tr(V'(G(V)-rho*H(V))V becomes minimal

    elseif maxit2 == 0 %decreasing SCF-Iter. number
        if k == 1
            it = 128; %start value with falling SCF iter. number
        elseif mod(k,floor(maxit1/4)) == 0
            it = floor(1/2*it); %halve the number after every quarter
        end
        [V,~,~] = scfiter(D,I,p,tol,it,V,shift,'smallest'); %determine V such that tr(V'(G(V)-rho*H(V))V becomes minimal

    elseif maxit2 == -1 %increasing SCF-Iter. number
        if k == 1
            it = 8; %start value with increasing SCF iter. number
        elseif mod(k,floor(maxit1/4)) == 0
            it = 2*it; %double the number after every quarter
        end
        [V,~,~] = scfiter(D,I,p,tol,it,V,shift,'smallest'); %determine V such that tr(V'(G(V)-rho*H(V))V becomes minimal
    end

    if k > 1 && abs(val(k) - val(k-1))/abs(val(k))< tol %Abortion criterion
        break;
    end

end
val = val(1:k);
val = [val, trace(V'*EvalG(V)*V)/trace(V'*EvalH(V)*V)];
