/* ----------------------------------------------------------------------
 Extension to the NUFEB package - A LAMMPS user package for Individual-based Modelling of Microbial Communities (https://github.com/nufeb/NUFEB/)
   
 Detailed description of the modifications can be found in:
 "Individual based modelling of biofilm colonisation"
 Marlis Reiber (Leibniz Universität Hannover)
 GAMM Archive for Students (GAMMAS), 3(1):1-14, 2022
------------------------------------------------------------------------- */

#include "fix_planktonic.h"

#include <math.h>
#include <string.h>

#include "atom.h"
#include "error.h"
#include "force.h"
#include "pointers.h"
#include "update.h"
#include "atom_vec_bio.h"
#include "comm.h"
#include "neigh_list.h"
#include "neighbor.h"
#include "neigh_request.h"
#include "random_mars.h"
#include "math_const.h"

using namespace LAMMPS_NS;
using namespace FixConst;
using namespace MathConst;

enum{XPLANE=0,YPLANE=1,ZPLANE=2};    // XYZ PLANE need to be 0,1,2

#define BIG 1.0e20

/* ---------------------------------------------------------------------- */

FixPlanktonic::FixPlanktonic(LAMMPS *lmp, int narg, char **arg) :
  Fix(lmp, narg, arg)
{
  avec = (AtomVecBio *) atom->style_match("bio");
  if (!avec) error->all(FLERR,"Fix planktonic requires atom style bio");

  if (narg != 9) error->all(FLERR,"Illegal fix planktonic command");

  // wallstyle args

  int n = strlen(&arg[3][2]) + 1;
  var = new char[n];
  strcpy(var,&arg[3][2]);

  int iarg = 4;
  if (strcmp(arg[iarg],"xplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix planktonic command");
    wallstyle = XPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  } else if (strcmp(arg[iarg],"yplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix planktonic command");
    wallstyle = YPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  } else if (strcmp(arg[iarg],"zplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix planktonic command");
    wallstyle = ZPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  }
  
  // initialize Marsaglia RNG with processor-unique seed
  seed = force->inumeric(FLERR,arg[3]);
  rng = new RanMars(lmp, seed + comm->me);
  // set maximum distance and distance to wall
  max_dist = force->numeric(FLERR,arg[7]);
  wall_dist = force->numeric(FLERR,arg[8]);

}

/* ---------------------------------------------------------------------- */

FixPlanktonic::~FixPlanktonic()
{
  delete [] var;
}

/* ---------------------------------------------------------------------- */

int FixPlanktonic::setmask()
{
  int mask = 0;
  mask |= POST_FORCE;
  // mask |= POST_FORCE_RESPA;
  return mask;
}

/* ---------------------------------------------------------------------- */

void FixPlanktonic::init()
{
  int irequest = neighbor->request((void *) this);
  neighbor->requests[irequest]->pair = 0;
  neighbor->requests[irequest]->fix = 1;
  neighbor->requests[irequest]->half = 0;
  neighbor->requests[irequest]->full = 1;
  dt = update->dt;
}

void FixPlanktonic::init_list(int id, NeighList *ptr)
{
	list = ptr;
}

/* ---------------------------------------------------------------------- */

void FixPlanktonic::post_force(int vflag)
{
  double dx,dy,dz,del1,del2,r,d_x,d_y,d_z;

  // set position of wall to initial settings

  double wlo = lo;
  double whi = hi;

  // loop over all my atoms
  // rsq = distance from wall
  // dx,dy,dz = signed distance from wall
  
  double **x = atom->x;
  double **f = atom->f;
  double *outer_radius = avec->outer_radius;
  int *type = atom->type;
  int *mask = atom->mask;
  int nlocal = atom->nlocal;

  int i,j,jj,inum,jnum,itype,jtype;
  int *ilist,*jlist,*numneigh,**firstneigh;
  inum = list->inum;
  ilist = list->ilist;
  numneigh = list->numneigh;
  firstneigh = list->firstneigh;
  
  // loop over all atoms  
  for (i = 0; i < atom->nlocal; i++) {
    itype = type[i];
    jlist = firstneigh[i];
    jnum = numneigh[i];
	  int biofilm = 0;
	
    if (wallstyle == XPLANE) {
      del1 = x[i][0] - wlo;
      del2 = whi - x[i][0];
      if (del1 < del2) dx = del1;
      else dx = -del2;
      r = abs(dx);
    } else if (wallstyle == YPLANE) {
      del1 = x[i][1] - wlo;
      del2 = whi - x[i][1];
      if (del1 < del2) dy = del1;
      else dy = -del2;
      r = abs(dy);
    } else if (wallstyle == ZPLANE) {
      del1 = x[i][2] - wlo;
      del2 = whi - x[i][2];
      if (del1 < del2) dz = del1;
      else dz = -del2;
      r = abs(dz);
    }
    
    // check if no neighbours, type = planktonic, not neighbouring wall
    if ((jnum == 0) && ( (r-outer_radius[i]) > wall_dist) && (itype == 2)) {
      double r1 = cos(rng->uniform()*2*MY_PI);
      double r2 = cos(rng->uniform()*2*MY_PI);
      double r3 = cos(rng->uniform()*2*MY_PI);
      double distance = rng->uniform();
      double magnitude = sqrt(r1*r1+r2*r2+r3*r3);
      
      // random movement  
      d_x = distance * max_dist * r1 / magnitude;
      d_y = distance * max_dist * r2 / magnitude;
      d_z = distance * max_dist * r3 / magnitude;
      x[i][0] += d_x;
      x[i][1] += d_y;
      x[i][2] += d_z;
      
    } else if ((itype == 2) && (jnum != 0) && ( (r-outer_radius[i]) > wall_dist)){
      //loop over neighbours of atom and check if planktonic, if a neighbour is not planktonic, change to biofilm type
      for (jj = 0; jj < jnum; jj++) {
        j = jlist[jj];
        j &= NEIGHMASK;

        jtype = type[j];
        
        if ((jtype == 1) && (biofilm == 0)) { 
          atom->type[i] = 1; //change type
          atom->mask[i] = 3; //change group
          biofilm = 1;
        }          
      }
      if ((biofilm == 0) ) {
        double r1 = cos(rng->uniform()*2*MY_PI);
        double r2 = cos(rng->uniform()*2*MY_PI);
        double r3 = cos(rng->uniform()*2*MY_PI);
        double distance = rng->uniform();
        double magnitude = sqrt(r1*r1+r2*r2+r3*r3);
      
        // random movement  
        d_x = distance * max_dist * r1 / magnitude;
        d_y = distance * max_dist * r2 / magnitude;
        d_z = distance * max_dist * r3 / magnitude;

        x[i][0] += d_x;
        x[i][1] += d_y;
        x[i][2] += d_z;
      }
    // check if neighbouring wall, change type to biofilm type
    } else if ((itype == 2 ) && ( (r-outer_radius[i]) <= wall_dist)){
      atom->type[i] = 1; //change type
      atom->mask[i] = 3; //change group
    }
  }
}

/* ----------------------------------------------------------------------
   maxsize of any atom's restart data
------------------------------------------------------------------------- */

int FixPlanktonic::maxsize_restart()
{
  return 4;
}

/* ----------------------------------------------------------------------
   size of atom nlocal's restart data
------------------------------------------------------------------------- */

int FixPlanktonic::size_restart(int nlocal)
{
  return 4;
}

/* ---------------------------------------------------------------------- */

void FixPlanktonic::reset_dt()
{
  dt = update->dt;
}