/* ----------------------------------------------------------------------
 Extension to the NUFEB package - A LAMMPS user package for Individual-based Modelling of Microbial Communities (https://github.com/nufeb/NUFEB/)
   
 Detailed description of the modifications can be found in:
 "Individual based modelling of biofilm colonisation"
 Marlis Reiber (Leibniz Universität Hannover)
 GAMM Archive for Students (GAMMAS), 3(1):1-14, 2022
------------------------------------------------------------------------- */

#include "fix_pair_table.h"

#include <math.h>
#include <string.h>

#include "atom.h"
#include "error.h"
#include "force.h"
#include "pointers.h"
#include "update.h"
#include "atom_vec_bio.h"
#include "memory.h"
#include "utils.h"
#include "comm.h"
#include "neigh_list.h"
#include "neighbor.h"
#include "neigh_request.h"

using namespace LAMMPS_NS;
using namespace FixConst;

enum{XPLANE=0,YPLANE=1,ZPLANE=2};    // XYZ PLANE need to be 0,1,2

#define BIG 1.0e20
#define MAXLINE 1024

/* ---------------------------------------------------------------------- */

FixPairTab::FixPairTab(LAMMPS *lmp, int narg, char **arg) :
  Fix(lmp, narg, arg)
{
  avec = (AtomVecBio *) atom->style_match("bio");
  if (!avec) error->all(FLERR,"Fix pairtab requires atom style bio");

  if (narg != 7) error->all(FLERR,"Illegal fix pairtab command");

  // particle coefficients
  
  if (strcmp(arg[3],"linear") == 0) tabstyle = LINEAR;
  else error->all(FLERR,"Unknown table style in pair_style command");
  
  tablength = force->inumeric(FLERR,arg[5]);
  if (tablength < 2) error->all(FLERR,"Illegal number of pair/table entries");

  ntables = 0;
  tables = NULL;
  
  int me;
  MPI_Comm_rank(world,&me);
  tables = (Table *)
    memory->srealloc(tables,(ntables+1)*sizeof(Table),"pair:tables");
  Table *tb = &tables[ntables];
  null_table(tb);
  if (me == 0) read_table(tb,arg[4],arg[6]);
  bcast_table(tb);
  
  // error check on table parameters
  // ensure cutoff is within table

  double rlo,rhi;
  if (tb->ninput <= 1) error->one(FLERR,"Invalid pair/table length");
  
  rlo = tb->rlo;
  rhi = tb->rhi;
  
  if (rlo >= rhi) error->all(FLERR,"pair/table values are not increasing");
  
  if (rlo <= 0.0) error->all(FLERR,"Invalid pair table cutoff");
  
}

/* ---------------------------------------------------------------------- */

FixPairTab::~FixPairTab()
{
  for (int m = 0; m < ntables; m++) free_table(&tables[m]);
  memory->sfree(tables);
}

/* ---------------------------------------------------------------------- */

int FixPairTab::setmask()
{
  int mask = 0;
  mask |= POST_FORCE;
  // mask |= POST_FORCE_RESPA;
  return mask;
}

/* ---------------------------------------------------------------------- */

void FixPairTab::init()
{
  int irequest = neighbor->request((void *) this);
  neighbor->requests[irequest]->pair = 0;
  neighbor->requests[irequest]->fix = 1;
  neighbor->requests[irequest]->half = 0;
  neighbor->requests[irequest]->full = 1;
  
  dt = update->dt;
}

void FixPairTab::init_list(int id, NeighList *ptr)
{
	list = ptr;
}

/* ---------------------------------------------------------------------- */

void FixPairTab::post_force(int vflag)
{
  double dx,dy,dz,rsq,r;

  // loop over all my atoms
  // rsq = distance from wall
  // dx,dy,dz = signed distance
  // compute force

  double **x = atom->x;
  double **f = atom->f;
  double **v = atom->v;
  double *outer_radius = avec->outer_radius;
  int *type = atom->type;
  int *mask = atom->mask;
  int nlocal = atom->nlocal;
  double delta = 0.0;
  
  double xtmp,ytmp,ztmp,delx,dely,delz,evdwl;
  double value;
  Table *tb;
  
  int tlm1 = tablength - 1;
  char estr[128];
  evdwl = 0.0;
  double counter;
  int overlap;
  int i,j,ii,jj,inum,jnum,itype,jtype,itable;
  
  int *ilist,*jlist,*numneigh,**firstneigh;
  inum = list->inum;
  ilist = list->ilist;
  numneigh = list->numneigh;
  firstneigh = list->firstneigh;
  double factor,n_x,n_y,n_z,p_x,p_y,p_z,f_x,f_y,f_z, m_f;
  
  tb = &tables[ntables];
  
  // loop over all atoms with neighbours
  for (ii = 0; ii < inum; ii++) {
    i = ilist[ii];
    xtmp = x[i][0];
    ytmp = x[i][1];
    ztmp = x[i][2];
    itype = type[i];
    jlist = firstneigh[i];
    jnum = numneigh[i];
	  overlap = 0;
    p_x = p_y = p_z = 0.0;
    f_x = f_y = f_z = 0.0;
    n_x = n_y = n_z = 0.0;
	  // loop over neighbours of atom i
    for (jj = 0; jj < jnum; jj++) {
      j = jlist[jj];
      j &= NEIGHMASK;

      delx = xtmp - x[j][0];
      dely = ytmp - x[j][1];
      delz = ztmp - x[j][2];
      rsq = delx*delx + dely*dely + delz*delz;
      jtype = type[j];
	    r = sqrt(rsq);

      if (tabstyle == LINEAR) {
	      // find minimal distance in table	
        itable = 0;
	      delta = abs((r - outer_radius[i] - outer_radius[j]) - tb->rfile[0]);
	      for(int jd = 1; jd < tb->ninput-1; jd++){
		    counter =  abs((r - outer_radius[i] - outer_radius[j]) - tb->rfile[jd]);
          if (counter < delta){
            delta = counter;
            itable = jd;			
          }
	      }
	      // calculate force value for corresponding distance
	      if (((abs(r - outer_radius[i] - outer_radius[j] - tb->rfile[itable+1]) >= abs(r - outer_radius[i] - outer_radius[j] - tb->rfile[itable-1])) && (itable != 0)) || (itable == tb->ninput-1)) { 
		      value = ((tb->ffile[itable] - tb->ffile[itable-1]) / abs(tb->rfile[itable] - tb->rfile[itable-1])) * abs(r - outer_radius[i] - outer_radius[j] - tb->rfile[itable-1]) + tb->ffile[itable-1];
	      } else {
	        value = ((tb->ffile[itable+1] - tb->ffile[itable]) / abs(tb->rfile[itable+1] - tb->rfile[itable])) * abs(r - outer_radius[i] - outer_radius[j] - tb->rfile[itable]) + tb->ffile[itable];
	      }
      }
	    // unit vector pointing from current atom to neighbour
	    n_x = -delx / r;
	    n_y = -dely / r;
	    n_z = -delz / r;
	    // unit vector total force
      m_f = sqrt(f[i][0]*f[i][0]+f[i][1]*f[i][1]+f[i][2]*f[i][2]);
	    f_x = f[i][0] / m_f;
	    f_y = f[i][1] / m_f;
	    f_z = f[i][2] / m_f;
	    // projected vector
	    factor = f_x*n_x + f_y*n_y + f_z*n_z;
	    // add to adhesion force: -table value * projected value
	    p_x += -value * factor * f_x;
	    p_y += -value * factor * f_y;
	    p_z += -value * factor * f_z;
	    // check if cells overlap
      if ((r < outer_radius[i] + outer_radius[j]) && (overlap == 0)){
		    overlap = 1;
      }
    }
	  // if overall force is larger than adhesion force, subtract adhesion force
	  // else set overall force to zero
	  // only if particles do not overlap
	  if ((m_f > sqrt(p_x*p_x+p_y*p_y+p_z*p_z)) && (jnum != 0) && (overlap == 0)){	
		  f[i][0] += p_x;
		  f[i][1] += p_y;
		  f[i][2] += p_z;		
	  } else if ((jnum != 0) && (overlap == 0)) {	
		  f[i][0] = 0;
		  f[i][1] = 0;
		  f[i][2] = 0;		
	  }
  }
}

/* ----------------------------------------------------------------------
   maxsize of any atom's restart data
------------------------------------------------------------------------- */

int FixPairTab::maxsize_restart()
{
  return 4;
}

/* ----------------------------------------------------------------------
   size of atom nlocal's restart data
------------------------------------------------------------------------- */

int FixPairTab::size_restart(int nlocal)
{
  return 4;
}

/* ---------------------------------------------------------------------- */

void FixPairTab::reset_dt()
{
  dt = update->dt;
}

void FixPairTab::null_table(Table *tb)
{
  tb->rfile = tb->efile = tb->ffile= NULL;
}

/* ----------------------------------------------------------------------
   read a table section from a tabulated potential file
   only called by proc 0
   this function sets these values in Table:
     ninput,rfile,efile,ffile,rflag,rlo,rhi
------------------------------------------------------------------------- */

void FixPairTab::read_table(Table *tb, char *file, char *keyword)
{
  char line[MAXLINE];

  // open file

  FILE *fp = force->open_potential(file);
  if (fp == NULL) {
    std::string str("Cannot open file ");
    str += file;
    error->one(FLERR,str.c_str());
  }

  // loop until section found with matching keyword

  while (1) {
    if (fgets(line,MAXLINE,fp) == NULL)
      error->one(FLERR,"Did not find keyword in table file");
    if (strspn(line," \t\n\r") == strlen(line)) continue;  // blank line
    if (line[0] == '#') continue;                          // comment
    char *word = strtok(line," \t\n\r");
    if (strcmp(word,keyword) == 0) break;            // matching keyword
    utils::sfgets(FLERR,line,MAXLINE,fp,file,error); // no match, skip section
    param_extract(tb,line);
    utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
    for (int i = 0; i < tb->ninput; i++)
      utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  }

  // read args on 2nd line of section
  // allocate table arrays for file values

  utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  param_extract(tb,line);
  memory->create(tb->rfile,tb->ninput,"pair:rfile");
  memory->create(tb->efile,tb->ninput,"pair:efile");
  memory->create(tb->ffile,tb->ninput,"pair:ffile");

  // read r,e,f table values from file

  int itmp;
  int cerror = 0;

  utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  for (int i = 0; i < tb->ninput; i++) {
    if (NULL == fgets(line,MAXLINE,fp))
      error->one(FLERR,"Premature end of file in pair table");
    if (4 != sscanf(line,"%d %lg %lg %lg",
                    &itmp,&tb->rfile[i],&tb->efile[i],&tb->ffile[i]))  ++cerror;
  }
  tb->rlo = tb->rfile[0];
  tb->rhi = tb->rfile[tb->ninput-1];	
  // close file

  fclose(fp);

  // warn if data was read incompletely, e.g. columns were missing

  if (cerror) {
    char str[128];
    sprintf(str,"%d of %d lines in table were incomplete\n"
            "  or could not be parsed completely",cerror,tb->ninput);
    error->warning(FLERR,str);
  }
}
 
/* ----------------------------------------------------------------------
   broadcast read-in table info from proc 0 to other procs
   this function communicates these values in Table:
     ninput,rfile,efile,ffile
------------------------------------------------------------------------- */

void FixPairTab::bcast_table(Table *tb)
{
  MPI_Bcast(&tb->ninput,1,MPI_INT,0,world);

  int me;
  MPI_Comm_rank(world,&me);
  if (me > 0) {
    memory->create(tb->rfile,tb->ninput,"pair:rfile");
    memory->create(tb->efile,tb->ninput,"pair:efile");
    memory->create(tb->ffile,tb->ninput,"pair:ffile");
  }

  MPI_Bcast(tb->rfile,tb->ninput,MPI_DOUBLE,0,world);
  MPI_Bcast(tb->efile,tb->ninput,MPI_DOUBLE,0,world);
  MPI_Bcast(tb->ffile,tb->ninput,MPI_DOUBLE,0,world);

}

/* ----------------------------------------------------------------------
   free all arrays in a table
------------------------------------------------------------------------- */

void FixPairTab::free_table(Table *tb)
{
  memory->destroy(tb->rfile);
  memory->destroy(tb->efile);
  memory->destroy(tb->ffile);
}

/* ----------------------------------------------------------------------
   extract attributes from parameter line in table section
   format of line: N value
   N is required, other params are optional
------------------------------------------------------------------------- */

void FixPairTab::param_extract(Table *tb, char *line)
{
  tb->ninput = 0;

  char *word = strtok(line," \t\n\r\f");
  while (word) {
    if (strcmp(word,"N") == 0) {
      word = strtok(NULL," \t\n\r\f");
      tb->ninput = atoi(word);
    } else {
      error->one(FLERR,"Invalid keyword in fix pair/table parameters");
    }
    word = strtok(NULL," \t\n\r\f");
  }

  if (tb->ninput == 0) error->one(FLERR,"fix pair/table parameters did not set N");
}