/* ----------------------------------------------------------------------
 Extension to the NUFEB package - A LAMMPS user package for Individual-based Modelling of Microbial Communities (https://github.com/nufeb/NUFEB/)
 
 Detailed description of the modifications can be found in:
 "Individual based modelling of biofilm colonisation"
 Marlis Reiber (Leibniz Universität Hannover)
 GAMM Archive for Students (GAMMAS), 3(1):1-14, 2022
------------------------------------------------------------------------- */

#include "fix_wall_table.h"

#include <math.h>
#include <string.h>

#include "atom.h"
#include "error.h"
#include "force.h"
#include "input.h"
#include "pointers.h"
#include "update.h"
#include "atom_vec_bio.h"
#include "memory.h"
#include "utils.h"

using namespace LAMMPS_NS;
using namespace FixConst;

enum{XPLANE=0,YPLANE=1,ZPLANE=2};    // XYZ PLANE need to be 0,1,2

#define BIG 1.0e20
#define MAXLINE 1024

/* ---------------------------------------------------------------------- */

FixWallTab::FixWallTab(LAMMPS *lmp, int narg, char **arg) :
  Fix(lmp, narg, arg)
{
  avec = (AtomVecBio *) atom->style_match("bio");
  if (!avec) error->all(FLERR,"Fix walladh requires atom style bio");

  if (narg != 10) error->all(FLERR,"Illegal fix walladh command");

  // wallstyle args

  int n = strlen(&arg[3][2]) + 1;
  var = new char[n];
  strcpy(var,&arg[3][2]);

  int iarg = 4;
  if (strcmp(arg[iarg],"xplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix walladh command");
    wallstyle = XPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  } else if (strcmp(arg[iarg],"yplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix walladh command");
    wallstyle = YPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  } else if (strcmp(arg[iarg],"zplane") == 0) {
    if (narg < iarg+3) error->all(FLERR,"Illegal fix walladh command");
    wallstyle = ZPLANE;
    if (strcmp(arg[iarg+1],"NULL") == 0) lo = -BIG;
    else lo = force->numeric(FLERR,arg[iarg+1]);
    if (strcmp(arg[iarg+2],"NULL") == 0) hi = BIG;
    else hi = force->numeric(FLERR,arg[iarg+2]);
    iarg += 3;
  }
  
  if (strcmp(arg[3],"linear") == 0) tabstyle = LINEAR;
  else error->all(FLERR,"Unknown table style in pair_style command");
  
  tablength = force->inumeric(FLERR,arg[8]);
  if (tablength < 2) error->all(FLERR,"Illegal number of wall/table entries");

  ntables = 0;
  tables = NULL;
  
  int me;
  MPI_Comm_rank(world,&me);
  tables = (Table *)
    memory->srealloc(tables,(ntables+1)*sizeof(Table),"wall:tables");
  Table *tb = &tables[ntables];
  null_table(tb);
  if (me == 0) read_table(tb,arg[7],arg[9]);
  bcast_table(tb);
  
  // error check on table parameters

  double rlo,rhi;
  if (tb->ninput <= 1) error->one(FLERR,"Invalid wall/table length");
  
  rlo = tb->rlo;
  rhi = tb->rhi;
  
  if (rlo >= rhi) error->all(FLERR,"wall/table values are not increasing");
  
  if (rlo <= 0.0) error->all(FLERR,"Invalid wall table cutoff");

}

/* ---------------------------------------------------------------------- */

FixWallTab::~FixWallTab()
{
  for (int m = 0; m < ntables; m++) free_table(&tables[m]);
  memory->sfree(tables);
  
  delete [] var;
}

/* ---------------------------------------------------------------------- */

int FixWallTab::setmask()
{
  int mask = 0;
  mask |= POST_FORCE;
  // mask |= POST_FORCE_RESPA;
  return mask;
}

/* ---------------------------------------------------------------------- */

void FixWallTab::init()
{
  dt = update->dt;
}

/* ---------------------------------------------------------------------- */

void FixWallTab::post_force(int vflag)
{
  double dx,dy,dz,del1,del2,r;

  // set position of wall to initial settings

  double wlo = lo;
  double whi = hi;

  // loop over all my atoms
  // r = distance from wall
  // dx,dy,dz = signed distance from wall
  // compute force 

  double **x = atom->x;
  double **f = atom->f;
  double *outer_radius = avec->outer_radius;
  int *type = atom->type;
  int *mask = atom->mask;
  int nlocal = atom->nlocal;

  double delta = 0.0;
  double value;
  Table *tb;
  int itable;  
  int *withdraw = avec->withdraw;
  
  for (int i = 0; i < nlocal; i++) {

    if   ((mask[i] & groupbit) > 0) {
      
      dx = dy = dz = 0.0;
    
      if (wallstyle == XPLANE) {
        del1 = x[i][0] - wlo;
        del2 = whi - x[i][0];
        if (del1 < del2) dx = del1;
        else dx = -del2;
        r = abs(dx);
      } else if (wallstyle == YPLANE) {
        del1 = x[i][1] - wlo;
        del2 = whi - x[i][1];
        if (del1 < del2) dy = del1;
        else dy = -del2;
        r = abs(dy);
      } else if (wallstyle == ZPLANE) {
        del1 = x[i][2] - wlo;
        del2 = whi - x[i][2];
        if (del1 < del2) dz = del1;
        else dz = -del2;
        r = abs(dz);
      }
      
      tb = &tables[ntables];
      // check if adhesion to wall
      // if close to wall, set withdraw = 1
      // if attached before (withdraw = 1) and outside range of forces, set withdraw = 0
      if ((r - outer_radius[i]) <= tb->rlo){
        if (withdraw[i] == 0){
          withdraw[i] = 1;
        }
        if (del1<del2){
        x[i][wallstyle] = outer_radius[i] + wlo; // move cell to not penetrate into wall
        } else {
        x[i][wallstyle] = whi - outer_radius[i];
        }
      } else if ((r - outer_radius[i] > tb->rhi) && (withdraw[i] == 1)) {
        withdraw[i] = 0;		
      }
    
      if (withdraw[i] == 1) {		  
        delta = abs((r - outer_radius[i]) - tb->rfile[0]); // first distance
        itable = 0;
        // find minimal distance in table
        for(int j = 1; j < tb->ninput-1; j++){
          if (abs((r - outer_radius[i]) - tb->rfile[j]) < delta) {
            delta = abs((r - outer_radius[i]) - tb->rfile[j]);
            itable = j;
          }
        }
        // calculate force value for corresponding distance
        if (((abs((r - outer_radius[i]) - tb->rfile[itable+1]) >= abs((r - outer_radius[i]) - tb->rfile[itable-1])) && (itable != 0)) || (itable == tb->ninput-1)){ 
          value = ((tb->ffile[itable] - tb->ffile[itable-1]) / abs(tb->rfile[itable] - tb->rfile[itable-1])) * abs((r - outer_radius[i]) - tb->rfile[itable-1]) + tb->ffile[itable-1];
          } else {
          value = ((tb->ffile[itable+1] - tb->ffile[itable]) / abs(tb->rfile[itable+1] - tb->rfile[itable])) * abs((r - outer_radius[i]) - tb->rfile[itable]) + tb->ffile[itable];
          }
        // if total force larger than 0 and adhesion force smaller than total force, subtract adhesion force
        // else set total force to zero
        if ( (del1 < del2) && (f[i][wallstyle] > 0) && (abs(f[i][wallstyle]) > abs(value)) ) {
          f[i][wallstyle] += value;
        } else if ( (del1 >= del2) && (f[i][wallstyle] < 0) && (abs(f[i][wallstyle]) > abs(value)) ){
          f[i][wallstyle] -= value;
        } else {
          f[i][1] = 0;
          f[i][2] = 0;
          f[i][3] = 0;
        }
      }
    }
  }
}

/* ----------------------------------------------------------------------
   maxsize of any atom's restart data
------------------------------------------------------------------------- */

int FixWallTab::maxsize_restart()
{
  return 4;
}

/* ----------------------------------------------------------------------
   size of atom nlocal's restart data
------------------------------------------------------------------------- */

int FixWallTab::size_restart(int nlocal)
{
  return 4;
}

/* ---------------------------------------------------------------------- */

void FixWallTab::reset_dt()
{
  dt = update->dt;
}

void FixWallTab::null_table(Table *tb)
{
  tb->rfile = tb->efile = tb->ffile= NULL;
}

/* ----------------------------------------------------------------------
   read a table section from a tabulated potential file
   only called by proc 0
   this function sets these values in Table:
     ninput,rfile,efile,ffile,rlo,rhi, cut
------------------------------------------------------------------------- */

void FixWallTab::read_table(Table *tb, char *file, char *keyword)
{
  char line[MAXLINE];

  // open file

  FILE *fp = force->open_potential(file);
  if (fp == NULL) {
    std::string str("Cannot open file ");
    str += file;
    error->one(FLERR,str.c_str());
  }

  // loop until section found with matching keyword

  while (1) {
    if (fgets(line,MAXLINE,fp) == NULL)
      error->one(FLERR,"Did not find keyword in table file");
    if (strspn(line," \t\n\r") == strlen(line)) continue;  // blank line
    if (line[0] == '#') continue;                          // comment
    char *word = strtok(line," \t\n\r");
    if (strcmp(word,keyword) == 0) break;            // matching keyword
    utils::sfgets(FLERR,line,MAXLINE,fp,file,error); // no match, skip section
    param_extract(tb,line);
    utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
    for (int i = 0; i < tb->ninput; i++)
      utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  }

  // read args on 2nd line of section
  // allocate table arrays for file values

  utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  param_extract(tb,line);
  memory->create(tb->rfile,tb->ninput,"wall:rfile");
  memory->create(tb->efile,tb->ninput,"wall:efile");
  memory->create(tb->ffile,tb->ninput,"wall:ffile");

  // read r,e,f table values from file

  int itmp;
  int cerror = 0;

  utils::sfgets(FLERR,line,MAXLINE,fp,file,error);
  for (int i = 0; i < tb->ninput; i++) {
    if (NULL == fgets(line,MAXLINE,fp))
      error->one(FLERR,"Premature end of file in pair table");
    if (4 != sscanf(line,"%d %lg %lg %lg",
                    &itmp,&tb->rfile[i],&tb->efile[i],&tb->ffile[i]))  ++cerror;
  }
  tb->rlo = tb->rfile[0];
  tb->rhi = tb->rfile[tb->ninput-1];
  
  // close file

  fclose(fp);

  // warn if data was read incompletely, e.g. columns were missing

  if (cerror) {
    char str[128];
    sprintf(str,"%d of %d lines in table were incomplete\n"
            "  or could not be parsed completely",cerror,tb->ninput);
    error->warning(FLERR,str);
  }
}
 
/* ----------------------------------------------------------------------
   broadcast read-in table info from proc 0 to other procs
   this function communicates these values in Table:
     ninput,rfile,efile,ffile
------------------------------------------------------------------------- */

void FixWallTab::bcast_table(Table *tb)
{
  MPI_Bcast(&tb->ninput,1,MPI_INT,0,world);

  int me;
  MPI_Comm_rank(world,&me);
  if (me > 0) {
    memory->create(tb->rfile,tb->ninput,"wall:rfile");
    memory->create(tb->efile,tb->ninput,"wall:efile");
    memory->create(tb->ffile,tb->ninput,"wall:ffile");
  }

  MPI_Bcast(tb->rfile,tb->ninput,MPI_DOUBLE,0,world);
  MPI_Bcast(tb->efile,tb->ninput,MPI_DOUBLE,0,world);
  MPI_Bcast(tb->ffile,tb->ninput,MPI_DOUBLE,0,world);

}

/* ----------------------------------------------------------------------
   free all arrays in a table
------------------------------------------------------------------------- */

void FixWallTab::free_table(Table *tb)
{
  memory->destroy(tb->rfile);
  memory->destroy(tb->efile);
  memory->destroy(tb->ffile);
}

/* ----------------------------------------------------------------------
   extract attributes from parameter line in table section
   format of line: N value
   N is required, other params are optional
------------------------------------------------------------------------- */

void FixWallTab::param_extract(Table *tb, char *line)
{
  tb->ninput = 0;

  char *word = strtok(line," \t\n\r\f");
  while (word) {
    if (strcmp(word,"N") == 0) {
      word = strtok(NULL," \t\n\r\f");
      tb->ninput = atoi(word);
    } else {
      error->one(FLERR,"Invalid keyword in fix wall/table parameters");
    }
    word = strtok(NULL," \t\n\r\f");
  }

  if (tb->ninput == 0) error->one(FLERR,"fix wall/table parameters did not set N");
}