%% Temporal error analysis of exponential integrators
% applied to the heat equation with dynamic boundary conditions
%
% This program computes a reference solution and coarser approximate
% solutions by three-stage exponential integrator scheme (expected
% convergence order 3) and plots ...
%   ... convergence history of the discrete L^\infty(0,T,||.||_A)-error
%       (with discrete energy norm ||.||_A).
%   ... initial data u(0) on domain
%   ... approximate solution u(T) on domain
%   ... evolution of u on dynamic part of boundary
%
% (spatial discretization by bilinear finite elements on a uniform mesh)
%
% Detailed descriptions of the approximation schemes and variables in
%  Error_ExpEuler.m

close all;
clear all;

%% parameters
Tend = .7;                      % time interval [0,Tend]
kRef = 10;                      % 2^kRef refinements of reference solution
NstepsRef = 2^kRef;             %  default kRef = 10;
tauRef = Tend / NstepsRef;

kappa = 0.02;                   % diffusion coefficient
alpha = 1;                      % parameter on boundary
beta = 0;                       % diffusion Laplace Beltrami
% rhsOmega,rhsGamma (right-hand sides - have to be defined in the code below)
% u0,p0 (initial data - has to be defined in the code below)
%
Plots2Tikz = false;             % Plots2Tikz = true or false
                                %  only set this to true, when matlab2tikz
                                %  package is imported
tic;
%% load matrices for h = 1/32
h = 1/2^5;                      % mesh size (= h)
N = (2^5+1)^2;                  % #(hats on domain including bdry)

loadMtx = matfile('saveMatrices.mat');
Ah = loadMtx.Ah;                % stiffness matrix of [0,1]^2
Mh = loadMtx.Mh;                % mass matrix of [0,1]^2
Eh = loadMtx.Eh;                % projection matrix to non-zero nodes (domain)
AhG = loadMtx.AhG;              % stiffness matrix of bdry of [0,1]^2
MhG = loadMtx.MhG;              % mass matrix of bdry of [0,1]^2
EhG = loadMtx.EhG;              % projection matrix to non-zero nodes (bdry)

%% matrix preparations
% mesh on Omega
Ah0 = Eh'*(Ah*Eh);              % stiffness matrix incl. Dirichlet data
Mh0 = Eh'*(Mh*Eh);              % mass matrix incl. Dirichlet data
n = size(Ah0,1);                % dof of u
% mesh on Gamma_dyn
MhG0 = EhG'*MhG*EhG;
AhG0 = EhG'*AhG*EhG;
m = size(MhG0,1);               % dof of p
dynNodes = 1+(1:m);             % corresponding nodes from Th
% concatenated matrices
MM = blkdiag(Mh0, MhG0);
AA = blkdiag(Ah0, alpha*MhG0+beta*AhG0);
BB = [MhG0, zeros(m, n-m), -MhG0];
SaddleMat = [AA, BB'; BB, zeros(m,m)];
M_expv = [MM, BB'; BB, zeros(m,m)];
A = @(x) [speye(n+m),zeros(n+m,m)]*(M_expv\[-AA*x;zeros(m,1)]);

%% initial data and right-hand sides
rhsOmega = Mh0*(Eh'*zeros(N,1));
rhsGamma = @(t) 3*cos(t*2*pi)*-EhG'*MhG*sin(2*pi*linspace(0,1,m+2)');
% coordinates of nodes on domain [0,1]^2 in lexographical order
xCoords = repmat(linspace(0,1,m+2),1,m+2)';
yCoords = reshape(repmat(linspace(0,1,m+2),m+2,1),[],1);
%
u0 = sin(xCoords*pi).*cos(yCoords*5*pi/2);
 
%% compute reference solution  (using three stage exponential integrator)
disp(['Computation of reference solution ', ...
    'by three-stage exponential integrator (Heun method) ...'])
uHeun = Eh'*u0;                  % the not-Dirichlet nodes
pHeun = u0(dynNodes);            % consistent ic

% 3rd order
uHeun_evo = zeros(n,NstepsRef+1);
uHeun_evo(:,1) = uHeun;
pHeun_evo = zeros(m,NstepsRef+1); 
pHeun_evo(:,1) = pHeun;

anorm = M_expv\[AA(:,round((n+m)/2));zeros(m,1)];
anorm = sum(abs(anorm(1:n)));

for j = 1 : NstepsRef
% step 0: compute B^-g_n,... (not necessary here)
% step 1: k_1
    % compute w_1, w_1_p
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1)*tauRef) ...
        - potenz_3(pHeun,h);zeros(m,1)]; % rhs = k_1
    w_1 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_1 / tauRef; zeros(m,1)];
    w_1_p = tmp(1:m+n);
% step 2: k_2
    % compute phi_{0,2/3}
    z0 = [uHeun; pHeun] - w_1;
    up_tmp = phipm( tauRef / 3, A, z0, 1e-7, false,10,anorm) + w_1;
    % compute w_2, w_2_p
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-2/3)*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_2
    w_2 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_2 / tauRef; zeros(m,1)];
    w_2_p = tmp(1:m+n);
% step 3: k_3
    % compute phi_{0,2/3}
    z0 = [uHeun; pHeun] - w_1 - 3*w_1_p + 3*w_2_p;
    up_tmp = phipm( tauRef * 2/3, A, z0, 1e-7, false, 10, anorm) ...
        - w_1 + 3*w_1_p + 2*w_2 - 3*w_2_p;
    % compute w_3, w_3_p, w_3_pp
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/3)*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_3
    w_3 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_3 / tauRef; zeros(m,1)];
    w_3_p = tmp(1:m+n);
% step 4: step u_n -> u_n+1
    % compute phi_{0,1}
    z0 = [uHeun; pHeun] - w_1 - 3/2*w_1_p + 3/2*w_3_p;
    up_tmp = phipm( tauRef, A, z0, 1e-7, false, 10, anorm ) ...
        - 1/2*w_1 + 3/2*w_1_p + 3/2*w_3 - 3/2*w_3_p;
    uHeun = up_tmp(1:n);
    pHeun = up_tmp((n+1):end);
    uHeun_evo(:,j+1) = uHeun;
    pHeun_evo(:,j+1) = pHeun;
end
disp(' ... completed.')
toc;
%% compute approximate solutions  (using three stage exponential integrator)
disp(['Computation of approximate solutions ', ...
    'by three-stage exponential integrator (Heun method) ...'])
tic;
A_ErrorMtx = zeros(1,kRef-1);
taus = zeros(1,kRef-1);
for k = 1:kRef-1
    tau = Tend / 2^k;
    Nsteps = 2^k;
    uHeun = Eh'*u0;                  % the not-Dirichlet nodes
    pHeun = u0(dynNodes);            % consistent ic

    tmpA_ErrorMtx = zeros(1,Nsteps);
    ratio = 2^(kRef-k);

    for j = 1 : Nsteps
    % step 0: compute B^-g_n,... (not necessary here)
    % step 1: k_1
        % compute w_1, w_1_p
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1)*tau) ...
            - potenz_3(pHeun,h);zeros(m,1)]; % rhs = k_1
        w_1 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_1 / tau; zeros(m,1)];
        w_1_p = tmp(1:m+n);
    % step 2: k_2
        % compute phi_{0,2/3}
        z0 = [uHeun; pHeun] - w_1;
        up_tmp = phipm( tau / 3, A, z0, 1e-7, false,10,anorm) + w_1;
        % compute w_2, w_2_p
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-2/3)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_2
        w_2 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_2 / tau; zeros(m,1)];
        w_2_p = tmp(1:m+n);
    % step 3: k_3
        % compute phi_{0,2/3}
        z0 = [uHeun; pHeun] - w_1 - 3*w_1_p + 3*w_2_p;
        up_tmp = phipm( tau * 2/3, A, z0, 1e-7, false, 10, anorm) ...
            - w_1 + 3*w_1_p + 2*w_2 - 3*w_2_p;
        % compute w_3, w_3_p, w_3_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/3)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_3
        w_3 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_3 / tau; zeros(m,1)];
        w_3_p = tmp(1:m+n);
    % step 4: step u_n -> u_n+1
        % compute phi_{0,1}
        z0 = [uHeun; pHeun] - w_1 - 3/2*w_1_p + 3/2*w_3_p;
        up_tmp = phipm( tau, A, z0, 1e-7, false, 10, anorm ) ...
            - 1/2*w_1 + 3/2*w_1_p + 3/2*w_3 - 3/2*w_3_p;
        uHeun = up_tmp(1:n);
        pHeun = up_tmp((n+1):end);
        
        % calculate energy error
        tmpVec = [uHeun - uHeun_evo(:,1+ratio*j); ...
            pHeun - pHeun_evo(:,1+ratio*j)];
        tmpA_ErrorMtx(j) = tmpVec'*AA*tmpVec;
    end
    % calculate L^\infty(0,T;||.||_A) error
    A_ErrorMtx(k) = sqrt(max(abs(tmpA_ErrorMtx)));
    taus(k) = Tend/2^k;
end
disp(' ... completed.')
toc;
%% error plot
disp('Assembling figures ...');
tic;
refLine = ones(1,kRef-1);
refLine1 = refLine .* taus;
refLine2 = refLine .* taus.^2;
refLine3 = refLine .* taus.^3;

figure();
loglog(taus,A_ErrorMtx,'-o','LineWidth',2,'MarkerSize',10, ...
    'Color',[0.9290, 0.6940, 0.1250], ...
    'DisplayName','third-order scheme (Heun)');
hold on
loglog(taus,refLine1,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau');
loglog(taus,refLine2,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau^2');
loglog(taus,refLine3,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau^3');

hold off
ax = gca; ax.XLim = [taus(end) taus(1)]; % ax.YLim = [1e-4 1e-1];
xlabel('step size \tau'); ylabel('energy error of x');
set(figure(1),'Color',[1 1 1]);
% legend('Location','southeast');
legend('third-order scheme (Heun)','reflines \tau, \tau^2, \tau^3', ...
    'Location','southeast');
grid on;
% saveas(figure(1),[pwd '/Figures/ThirdOrder/', ...
%     'Heun_Energy_Error_Time_kRef_',num2str(kRef),'.fig']);
if Plots2Tikz == true
    matlab2tikz('filename',[pwd '/Figures/ThirdOrder/', ...
        'Heun_Energy_Error_Time_kRef_',num2str(kRef),'.tex'], ...
        'height','0.45\textwidth','width','0.9\textwidth')
end

%% domain plot of u(0)
figure();
xCoordsPlt = reshape(xCoords,m+2,m+2);
yCoordsPlt = reshape(yCoords,m+2,m+2);
U = reshape(u0,m+2,m+2);
surf(xCoordsPlt,yCoordsPlt,U,'EdgeColor','None');
axis([0 1 0 1 -1 1]);
xlabel('x');
ylabel('y');
set(figure(2),'Color',[1 1 1]);
title('Plot of u(0) on whole domain');
colormap(1-winter);

%% domain plot of u(Tend)
figure();
U = reshape(Eh*uHeun_evo(:,end),m+2,m+2);
surf(xCoordsPlt,yCoordsPlt,U,'EdgeColor','None');
axis([0 1 0 1 -1 1]);
set(figure(3),'Color',[1 1 1]);
xlabel('x'); 
ylabel('y');
title('Plot of u(T) on whole domain');
colormap(1-winter);
% saveas(figure(3),[pwd '/Figures/ThirdOrder/','Heun_uT_kRef_',...
%     num2str(kRef),'.fig']);

%% evolution plot of dynamic boundary
figure();
plot(linspace(0,1,m+2), [0;pHeun_evo(:,1);0], '--', 'LineWidth', 1);
hold on;
if kRef > 3 % set 16 snapshots if possible
    nSnap = 2^4; ratio = 2^(kRef-4);
else
    nSnap = 2^kRef; ratio = 1;
end
for i = 1+ratio : ratio : NstepsRef+1
    ColTmp = [0.6350, 0.0780, 0.1840] + ...
        [1-0.6350, 1-0.0780, 1-0.1840]*(NstepsRef-i-1)/NstepsRef;
    plot(linspace(0,1,m+2), [0;pHeun_evo(:,i);0], '-', ...
        'LineWidth', 1, 'Color', ColTmp);
end
plot(linspace(0,1,m+2), [0;pHeun_evo(:,end);0],'-',...
    'LineWidth',2,'Color',[0.6350, 0.0780, 0.1840]);
hold off
axis([0 1 -0.4 1.1]);
grid on;
set(figure(4),'Color',[1 1 1]);
xlabel('x');
title('Evolution of function on boundary \Gamma_{dyn}');
% saveas(figure(4),[pwd '/Figures/ThirdOrder/','Heun_Evolution_kRef_',...
%     num2str(kRef),'.fig']);
if Plots2Tikz == true
    cleanfigure
    matlab2tikz('filename',[pwd '/Figures/ThirdOrder/', ...
        'Heun_Evolution_kRef_',num2str(kRef),'.tex'], ...
        'height','0.3\textwidth','width','0.5\textwidth')
end

disp(' ... completed.')
toc;