%% Temporal error analysis of exponential integrators
% applied to the heat equation with dynamic boundary conditions
%
% This program computes a reference solution by a five-stage exponential
% integrator scheme and approximate solutions by ...
%   ... exponential Euler scheme (convergence order one assured by theory)
%   ... two-stage scheme (order 3/2 assured by theory, 2 expected due to regularity)
%   ... three-stage scheme (order 3 expected)
%   ... five-stage scheme (order 4 expected)
% Plots convergence history of the discrete L^\infty(0,T,||.||_A)-error
% (with discrete energy norm ||.||_A).
%
% (spatial discretization by bilinear finite elements on a uniform mesh)
%
% Detailed descriptions of the approximation schemes and variables in
%  Error_ExpEuler.m

close all;
clear all;

%% parameters
Tend = .7;                      % time interval [0,Tend]
kRef = 10;                      % 2^kRef refinements of reference solution
NstepsRef = 2^kRef;             %  default kRef = 10;
tauRef = Tend / NstepsRef;
phiTol = 1e-8;                  % phiTol = 1e-7; tolerance of approximation
                                %  of phi-functions \phi_{0,c} of reference
                                %  solution
%
kappa = 0.02;                   % diffusion coefficient
alpha = 1;                      % parameter on boundary
beta = 0;                       % diffusion Laplace Beltrami
% rhsOmega,rhsGamma (right-hand sides - have to be defined in the code below)
% u0,p0 (initial data - has to be defined in the code below)
%
Plots2Tikz = false;             % Plots2Tikz = true or false
                                %  only set this to true, when matlab2tikz
                                %  package is imported
tic;
%% load matrices for h = 1/32
h = 1/2^5;                      % mesh size (= h)
N = (2^5+1)^2;                  % #(hats on domain including bdry)

loadMtx = matfile('saveMatrices.mat');
Ah = loadMtx.Ah;                % stiffness matrix of [0,1]^2
Mh = loadMtx.Mh;                % mass matrix of [0,1]^2
Eh = loadMtx.Eh;                % projection matrix to non-zero nodes (domain)
AhG = loadMtx.AhG;              % stiffness matrix of bdry of [0,1]^2
MhG = loadMtx.MhG;              % mass matrix of bdry of [0,1]^2
EhG = loadMtx.EhG;              % projection matrix to non-zero nodes (bdry)

%% matrix preparations
% mesh on Omega
Ah0 = Eh'*(Ah*Eh);              % stiffness matrix incl. Dirichlet data
Mh0 = Eh'*(Mh*Eh);              % mass matrix incl. Dirichlet data
n = size(Ah0,1);                % dof of u
% mesh on Gamma_dyn
MhG0 = EhG'*MhG*EhG;
AhG0 = EhG'*AhG*EhG;
m = size(MhG0,1);               % dof of p
dynNodes = 1+(1:m);             % corresponding nodes from mesh of domain
% concatenated matrices
MM = blkdiag(Mh0, MhG0);
AA = blkdiag(Ah0, alpha*MhG0+beta*AhG0);
BB = [MhG0, zeros(m, n-m), -MhG0];
SaddleMat = [AA, BB'; BB, zeros(m,m)];
M_expv = [MM, BB'; BB, zeros(m,m)];
A = @(x) [speye(n+m),zeros(n+m,m)]*(M_expv\[-AA*x;zeros(m,1)]);

%% initial data and right-hand sides
rhsOmega = Mh0*(Eh'*zeros(N,1));
rhsGamma = @(t) 3*cos(t*2*pi)*-EhG'*MhG*sin(2*pi*linspace(0,1,m+2)');
% coordinates of nodes on domain [0,1]^2 in lexographical order
xCoords = repmat(linspace(0,1,m+2),1,m+2)';
yCoords = reshape(repmat(linspace(0,1,m+2),m+2,1),[],1);
%
u0 = sin(xCoords*pi).*cos(yCoords*5*pi/2);
 
%% compute reference solution  (using fourth order scheme)
disp(['Computation of reference solution ', ...
    'by five-stage exponential integrator ...'])
uFour = Eh'*u0;                  % the not-Dirichlet nodes
pFour = u0(dynNodes);            % consistent ic

% fourth order
uFour_evo = zeros(n,NstepsRef+1);
uFour_evo(:,1) = uFour;
pFour_evo = zeros(m,NstepsRef+1); 
pFour_evo(:,1) = pFour;

anorm = M_expv\[AA(:,round((n+m)/2));zeros(m,1)];
anorm = sum(abs(anorm(1:n)));

for j = 1 : NstepsRef
% step 0: compute B^-g_n,... (not necessary here)
% step 1: k_1 (compute w_1, w_1_p, w_1_pp)
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1)*tauRef) ...
        - potenz_3(pFour,h);zeros(m,1)]; % rhs = k_1
    w_1 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_1 / tauRef; zeros(m,1)];
    w_1_p = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_1_p / tauRef; zeros(m,1)];
    w_1_pp = tmp(1:m+n);
% step 2: k_2
    % compute phi_{0,1/2}
    z0 = [uFour; pFour] - w_1;
    up_tmp = phipm( tauRef / 2, A, z0, phiTol, false, 10, anorm ) + w_1;
    % compute w_2, w_2_p, w_2_pp
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_2
    w_2 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_2 / tauRef; zeros(m,1)];
    w_2_p = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_2_p / tauRef; zeros(m,1)];
    w_2_pp = tmp(1:m+n);
% step 3: k_3
    % compute phi_{0,1/2}
    z0 = [uFour; pFour] - w_1 - 4*w_1_p + 4*w_2_p;
    up_tmp = phipm( tauRef / 2, A, z0, phiTol, false, 10, anorm ) ...
        - w_1 + 4*w_1_p + 2*w_2 - 4*w_2_p;
    % compute w_3, w_3_p, w_3_pp
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_3
    w_3 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_3 / tauRef; zeros(m,1)];
    w_3_p = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_3_p / tauRef; zeros(m,1)];
    w_3_pp = tmp(1:m+n);
% step 4: k_4
    % compute phi_{0,1}
    z0 = [uFour; pFour] - w_1 - 2*w_1_p + w_2_p + w_3_p;
    up_tmp = phipm( tauRef, A, z0, phiTol, false, 10, anorm ) ...
        - w_1 + 2*w_1_p + w_2 - w_2_p + w_3 - w_3_p;
    % compute w_4, w_4_p, w_4_pp
    tmp = SaddleMat \ [rhsOmega;rhsGamma(j*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_4
    w_4 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_4 / tauRef; zeros(m,1)];
    w_4_p = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_4_p / tauRef; zeros(m,1)];
    w_4_pp = tmp(1:m+n);
% step 5: k_5
    % compute phi_{0,1/2}
    z0 = [uFour; pFour] - w_1 - 3*w_1_p - 4*w_1_pp + 2*w_2_p + 4*w_2_pp ...
        + 2*w_3_p + 4*w_3_pp - w_4_p - 4*w_4_pp;
    phi_tmp = phipm( tauRef / 2, A, z0, phiTol, false, 10, anorm );
    % compute phi_{0,1}
    z0 = (- w_1_p - 4*w_1_pp + w_2_p + 4*w_2_pp + w_3_p + 4*w_3_pp ...
        - w_4_p - 4*w_4_pp) / 4;
    up_tmp = phi_tmp + phipm( tauRef, A, z0, phiTol, false, 10, anorm ) ...
        + ( w_1 + w_1_p + 20*w_1_pp + w_2 + 3*w_2_p - 20*w_2_pp ...
        + w_3 + 3*w_3_p - 20*w_3_pp + w_4 - 7*w_4_p + 20*w_4_pp ) / 4;
    % compute w_5, w_5_p, w_5_pp
    tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tauRef) ...
        - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_5
    w_5 = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_5 / tauRef; zeros(m,1)];
    w_5_p = tmp(1:m+n);
    tmp = SaddleMat \ [MM * w_5_p / tauRef; zeros(m,1)];
    w_5_pp = tmp(1:m+n);
% step 6: compute u_n+1
    % phi_{0,1}
    z0 = [uFour; pFour] - w_1 - 3*w_1_p - 4*w_1_pp - w_4_p - 4*w_4_pp ...
        + 4*w_5_p + 8*w_5_pp;
    up_tmp = phipm( tauRef, A, z0, phiTol, false, 10, anorm ) ...
        - w_1_p + 4*w_1_pp + w_4 - 3*w_4_p + 4*w_4_pp + 4*w_5_p - 8*w_5_pp;
    uFour = up_tmp(1:n);
    pFour = up_tmp((n+1):end);
    uFour_evo(:,j+1) = uFour;
    pFour_evo(:,j+1) = pFour;
end
disp(' ... completed.')
toc;
%% compute approximate solutions  (using Exponential Euler scheme)
disp(['Computation of approximate solutions ', ...
    'by first-order exponential integrator ...'])
tic;
A_ErrorMtx = zeros(4,kRef-1);
taus = zeros(1,kRef-1);
for k = 1:kRef-1
    tau = Tend / 2^k;
    Nsteps = 2^k;
    uEul = Eh'*u0;                  % the not-Dirichlet nodes
    pEul = u0(dynNodes);            % consistent ic

    tmpA_ErrorMtx = zeros(1,Nsteps);
    ratio = 2^(kRef-k);

    for j = 1 : Nsteps
        % step 1: compute B^-g_n,... (not necessary here)
        % step 2: compute w_n
        wnmun = SaddleMat\[rhsOmega;rhsGamma((j-1)*tau) ...
            - potenz_3(pEul,h);zeros(m,1)];
        wn = wnmun(1:n+m);
        % step 3: compute z by exponential
        z0 = [uEul; pEul] - wn;
        x_Eul = phipm( tau, A, z0, 1e-7, false, 10, anorm ) + wn;
        uEul = x_Eul(1:n);
        pEul = x_Eul(n+1:end);
        % calculate energy error
        tmpVec = [uEul - uFour_evo(:,1+ratio*j); ...
            pEul - pFour_evo(:,1+ratio*j)];
        tmpA_ErrorMtx(j) = tmpVec'*AA*tmpVec;
    end
    % calculate L^\infty(0,T;||.||_A) error
    A_ErrorMtx(1,k) = sqrt(max(abs(tmpA_ErrorMtx)));
    taus(k) = Tend/2^k;
end
disp(' ... completed.')
toc;
%% compute approximate solutions  (using second order scheme)
disp(['Computation of approximate solutions ', ...
    'by second-order exponential integrator ...'])
tic;
for k = 1:kRef-1
    tau = Tend / 2^k;
    Nsteps = 2^k;
    uSec = Eh'*u0;                  % the not-Dirichlet nodes
    pSec = u0(dynNodes);            % consistent ic

    tmpA_ErrorMtx = zeros(1,Nsteps);
    ratio = 2^(kRef-k);

    for j = 1 : Nsteps
        % step 1: compute B^-g_n,... (not necessary here)
        % step 2: compute w_n
        wnmun = SaddleMat\[rhsOmega;rhsGamma((j-1)*tau) ...
            - potenz_3(pSec,h);zeros(m,1)];
        wn = wnmun(1:n+m);
        % step 3: compute z by exponential
        z0 = [uSec; pSec] - wn;
        xSec_Eul = phipm( tau, A, z0, 1e-7, false, 10, anorm ) + wn;
        pSec_Eul = xSec_Eul((n+1):end);    
        % step 4: compute B^-\dot g_n,... (not necessary here) 
        % step 5: compute w'_n
        ww_p = SaddleMat \ ...
            [zeros(n,1);(rhsGamma(j*tau)-potenz_3(pSec_Eul,h) ...
            - rhsGamma((j-1)*tau)+potenz_3(pSec,h))/tau;zeros(m,1)]; 
        ww_p_n = ww_p(1:(n+m));    
        % step 6: compute w''_n
        ww_pp = SaddleMat \ [MM*ww_p_n;zeros(m,1)];
        ww_pp_n = ww_pp(1:(n+m));    
        % step 7: compute z by exponential 
        xSec = xSec_Eul + phipm( tau, A, ww_pp_n, 1e-7, false, 10, anorm ) ...
                    -ww_pp_n + tau*ww_p_n; % line (8)
        uSec = xSec(1:n);
        pSec = xSec((n+1):end);
        
        % calculate energy error
        tmpVec = [uSec - uFour_evo(:,1+ratio*j); ...
            pSec - pFour_evo(:,1+ratio*j)];
        tmpA_ErrorMtx(j) = tmpVec'*AA*tmpVec;
    end
    % calculate L^\infty(0,T;||.||_A) error
    A_ErrorMtx(2,k) = sqrt(max(abs(tmpA_ErrorMtx)));
end
disp(' ... completed.')
toc;
%% compute approximate solutions  (using three stage exponential integrator)
disp(['Computation of approximate solutions ', ...
    'by three-stage exponential integrator ...'])
tic;
for k = 1:kRef-1
    tau = Tend / 2^k;
    Nsteps = 2^k;
    u3rd = Eh'*u0;                  % the not-Dirichlet nodes
    p3rd = u0(dynNodes);            % consistent ic

    tmpA_ErrorMtx = zeros(1,Nsteps);
    ratio = 2^(kRef-k);

    for j = 1 : Nsteps
    % step 0: compute B^-g_n,... (not necessary here)
    % step 1: k_1
        % compute w_1, w_1_p
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1)*tau) ...
            - potenz_3(p3rd,h);zeros(m,1)]; % rhs = k_1
        w_1 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_1 / tau; zeros(m,1)];
        w_1_p = tmp(1:m+n);
    % step 2: k_2
        % compute phi_{0,2/3}
        z0 = [u3rd; p3rd] - w_1;
        up_tmp = phipm( tau * 2/3, A, z0, 1e-7, false, 10, anorm ) + w_1;
        % compute w_2, w_2_p
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/3)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_2
        w_2 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_2 / tau; zeros(m,1)];
        w_2_p = tmp(1:m+n);
    % step 3: k_3
        % compute phi_{0,2/3}
        z0 = [u3rd; p3rd] - w_1 - 3/2*w_1_p + 3/2*w_2_p;
        up_tmp = phipm( tau * 2/3, A, z0, 1e-7, false, 10, anorm ) ...
            + 3/2*w_1_p + w_2 - 3/2*w_2_p;
        % compute w_3, w_3_p, w_3_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/3)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_3
        w_3 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_3 / tau; zeros(m,1)];
        w_3_p = tmp(1:m+n);
    % step 4: step u_n -> u_n+1
        % compute phi_{0,1}
        z0 = [u3rd; p3rd] - w_1 - 3/2*w_1_p + 3/2*w_3_p;
        up_tmp = phipm( tau, A, z0, 1e-7, false, 10, anorm ) ...
            - 1/2*w_1 + 3/2*w_1_p + 3/2*w_3 - 3/2*w_3_p;
        u3rd = up_tmp(1:n);
        p3rd = up_tmp((n+1):end);
        
        % calculate energy error
        tmpVec = [u3rd - uFour_evo(:,1+ratio*j); ...
            p3rd - pFour_evo(:,1+ratio*j)];
        tmpA_ErrorMtx(j) = tmpVec'*AA*tmpVec;
    end
    % calculate L^\infty(0,T;||.||_A) error
    A_ErrorMtx(3,k) = sqrt(max(abs(tmpA_ErrorMtx)));
end
disp(' ... completed.')
toc;
%% compute fourth order approximate solutions
disp(['Computation of approximate solutions ', ...
    'by five-stage exponential integrator ...'])
tic;
for k = 1:kRef-1
    tau = Tend / 2^k;
    Nsteps = 2^k;
    uFour = Eh'*u0;                  % the not-Dirichlet nodes
    pFour = u0(dynNodes);            % consistent ic

    tmpA_ErrorMtx = zeros(1,Nsteps);
    ratio = 2^(kRef-k);

    for j = 1 : Nsteps
    % step 0: compute B^-g_n,... (not necessary here)
    % step 1: k_1 (compute w_1, w_1_p, w_1_pp)
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1)*tau) ...
            - potenz_3(pFour,h);zeros(m,1)]; % rhs = k_1
        w_1 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_1 / tau; zeros(m,1)];
        w_1_p = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_1_p / tau; zeros(m,1)];
        w_1_pp = tmp(1:m+n);
    % step 2: k_2
        % compute phi_{0,1/2}
        z0 = [uFour; pFour] - w_1;
        up_tmp = phipm( tau / 2, A, z0, phiTol, false, 10, anorm) + w_1;
        % compute w_2, w_2_p, w_2_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_2
        w_2 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_2 / tau; zeros(m,1)];
        w_2_p = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_2_p / tau; zeros(m,1)];
        w_2_pp = tmp(1:m+n);
    % step 3: k_3
        % compute phi_{0,1/2}
        z0 = [uFour; pFour] - w_1 - 4*w_1_p + 4*w_2_p;
        up_tmp = phipm( tau / 2, A, z0, phiTol, false, 10, anorm) ...
            - w_1 + 4*w_1_p + 2*w_2 - 4*w_2_p;
        % compute w_3, w_3_p, w_3_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_3
        w_3 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_3 / tau; zeros(m,1)];
        w_3_p = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_3_p / tau; zeros(m,1)];
        w_3_pp = tmp(1:m+n);
    % step 4: k_4
        % compute phi_{0,1}
        z0 = [uFour; pFour] - w_1 - 2*w_1_p + w_2_p + w_3_p;
        up_tmp = phipm( tau, A, z0, phiTol, false, 10, anorm ) ...
            - w_1 + 2*w_1_p + w_2 - w_2_p + w_3 - w_3_p;
        % compute w_4, w_4_p, w_4_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma(j*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_4
        w_4 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_4 / tau; zeros(m,1)];
        w_4_p = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_4_p / tau; zeros(m,1)];
        w_4_pp = tmp(1:m+n);
    % step 5: k_5
        % compute phi_{0,1/2}
        z0 = [uFour; pFour] - w_1 - 3*w_1_p - 4*w_1_pp ...
            + 2*w_2_p + 4*w_2_pp + 2*w_3_p + 4*w_3_pp - w_4_p - 4*w_4_pp;
        phi_tmp = phipm( tau / 2, A, z0, phiTol, false, 10, anorm );
        % compute phi_{0,1}
        z0 = (- w_1_p - 4*w_1_pp + w_2_p + 4*w_2_pp + w_3_p + 4*w_3_pp ...
            - w_4_p - 4*w_4_pp) / 4;
        up_tmp = phi_tmp + phipm( tau, A, z0, phiTol, false, 10, anorm ) ...
            + ( w_1 + w_1_p + 20*w_1_pp + w_2 + 3*w_2_p - 20*w_2_pp ...
            + w_3 + 3*w_3_p - 20*w_3_pp + w_4 - 7*w_4_p + 20*w_4_pp ) / 4;
        % compute w_5, w_5_p, w_5_pp
        tmp = SaddleMat \ [rhsOmega;rhsGamma((j-1/2)*tau) ...
            - potenz_3(up_tmp(n+1:end),h);zeros(m,1)]; % rhs = k_5
        w_5 = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_5 / tau; zeros(m,1)];
        w_5_p = tmp(1:m+n);
        tmp = SaddleMat \ [MM * w_5_p / tau; zeros(m,1)];
        w_5_pp = tmp(1:m+n);
    % step 6: compute u_n+1
        % phi_{0,1}
        z0 = [uFour; pFour] - w_1 - 3*w_1_p - 4*w_1_pp ...
            - w_4_p - 4*w_4_pp + 4*w_5_p + 8*w_5_pp;
        up_tmp = phipm( tau, A, z0, phiTol, false, 10, anorm ) ...
            - w_1_p + 4*w_1_pp + w_4 - 3*w_4_p + 4*w_4_pp + 4*w_5_p - 8*w_5_pp;
        uFour = up_tmp(1:n);
        pFour = up_tmp((n+1):end);
        
        % calculate energy error
        tmpVec = [uFour - uFour_evo(:,1+ratio*j); ...
            pFour - pFour_evo(:,1+ratio*j)];
        tmpA_ErrorMtx(j) = tmpVec'*AA*tmpVec;
    end
    % calculate L^\infty(0,T;||.||_A) error
    A_ErrorMtx(4,k) = sqrt(max(abs(tmpA_ErrorMtx)));
end
disp(' ... completed.')
toc;
%% error plot
disp('Assembling figures ...');
tic;
refLine = ones(1,kRef-1);
refLine1 = refLine .* taus;
refLine2 = refLine .* taus.^2;
refLine3 = refLine .* taus.^3;
refLine4 = refLine .* taus.^4;

% convergence history of temporal errors
figure();
loglog(taus,A_ErrorMtx(1,:),'-^','LineWidth',2,'MarkerSize',10, ...
    'Color',[0, 0.4470, 0.7410],'DisplayName','exponential Euler');
hold on
loglog(taus,A_ErrorMtx(2,:),'-s','LineWidth',2,'MarkerSize',10, ...
    'Color',[0.4660, 0.6740, 0.1880],'DisplayName','second-order scheme');
loglog(taus,A_ErrorMtx(3,:),'-o','LineWidth',2,'MarkerSize',10, ...
    'Color',[0.9290, 0.6940, 0.1250],'DisplayName','third-order scheme');
loglog(taus,A_ErrorMtx(4,:),'-x','LineWidth',2,'MarkerSize',10, ...
    'Color',[0.6350, 0.0780, 0.1840],'DisplayName','fourth-order scheme');
% plot reference lines
loglog(taus,refLine1,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau');
loglog(taus,refLine2,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau^2');
loglog(taus,refLine3,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau^3');
loglog(taus,refLine4,'--','LineWidth',1.5,'Color',[0.25, 0.25, 0.25], ...
    'DisplayName','reference line \tau^4');
hold off
ax = gca; ax.XLim = [taus(end) taus(1)]; ax.YLim = [1e-8 1];
xlabel('step size \tau'); ylabel('energy error of x');
set(figure(1),'Color',[1 1 1]);
% legend('Location','southeast');
legend('exponential Euler','second-order scheme','third-order scheme', ...
    'fourth-order scheme','reference lines \tau, \tau^2, \tau^3, \tau^4', ...
    'Location','southeast');
grid on;
% saveas(figure(1),[pwd '/Figures/','Temporal_Error_Analysis_kRef_',...
%     num2str(kRef),'.fig']);
if Plots2Tikz == true
    matlab2tikz('filename',[pwd '/Figures/', ...
        'Temporal_Error_Analysis_kRef_',num2str(kRef),'.tex'], ...
        'height','0.3\textwidth','width','0.9\textwidth')
end

%% domain plot of u(0)
figure();
xCoordsPlt = reshape(xCoords,m+2,m+2);
yCoordsPlt = reshape(yCoords,m+2,m+2);
U = reshape(u0,m+2,m+2);
surf(xCoordsPlt,yCoordsPlt,U,'EdgeColor','None');
axis([0 1 0 1 -1 1]);
set(figure(2),'Color',[1 1 1]);
xlabel('x');
ylabel('y');
title('Plot of u(0) on whole domain');
colormap(1-winter);
% saveas(figure(2),[pwd '/Figures/','FourthOrder_u0_kRef_',...
%     num2str(kRef),'.fig']);

%% domain plot of u(Tend)
figure();
U = reshape(Eh*uFour_evo(:,end),m+2,m+2);
surf(xCoordsPlt,yCoordsPlt,U,'EdgeColor','None');
axis([0 1 0 1 -1 1]);
set(figure(3),'Color',[1 1 1]);
xlabel('x'); 
ylabel('y');
title('Plot of u(T) on whole domain');
colormap(1-winter);
% saveas(figure(3),[pwd '/Figures/','FourthOrder_uT_kRef_',...
%     num2str(kRef),'.fig']);

%% evolution plot of dynamic boundary
figure();
plot(linspace(0,1,m+2), [0;pFour_evo(:,1);0], '--', 'LineWidth', 1);
hold on;
if kRef > 3 % set 16 snapshots if possible
    nSnap = 2^4; ratio = 2^(kRef-4);
else
    nSnap = 2^kRef; ratio = 1;
end
for i = 1+ratio : ratio : NstepsRef+1
    ColTmp = [0.6350, 0.0780, 0.1840] + ...
        [1-0.6350, 1-0.0780, 1-0.1840]*(NstepsRef-i-1)/NstepsRef;
    plot(linspace(0,1,m+2), [0;pFour_evo(:,i);0], '-', ...
        'LineWidth', 1, 'Color', ColTmp);
end
plot(linspace(0,1,m+2), [0;pFour_evo(:,end);0],'-',...
    'LineWidth',2,'Color',[0.6350, 0.0780, 0.1840]);
hold off
axis([0 1 -0.4 1.1]);
grid on;
set(figure(4),'Color',[1 1 1]);
xlabel('x');
title('Evolution of function on boundary \Gamma_{dyn}');
% saveas(figure(4),[pwd '/Figures/','FourthOrder_Evolution_kRef_',...
%     num2str(kRef),'.fig']);
if Plots2Tikz == true
    cleanfigure
    matlab2tikz('filename',[pwd '/Figures/', ...
        'FourthOrder_Evolution_kRef_',num2str(kRef),'.tex'], ...
        'height','0.3\textwidth','width','0.5\textwidth')
end

disp(' ... completed.')
toc;