'''
Module for numerical integration of systems of ordinary differential
equations.
'''
import numpy as np


def euler(f, t_list, x0):
    '''Euler method on equidistant mesh.

    This function approximately solves a system ordinary differential
    equations with an initial condition

        x'(t) = f(t, x(t))
        x(t_list[0]) = x0

    using the Euler method over the mesh t_list.

    Parameters
    ----------
    f : callable
        Right-hand side function. Takes a time point and a
        one-dimensional NumPy array as inputs, returns a one-dimensional
        NumPy array of the same size.
    t_list : list of floats
        Time mesh (t_list[0] is the initial time, t_list[-1] is the
        final time).
    x0 : One-dimensional NumPy array
        Initial condition.

    Returns
    -------
    x_sol :
        NumPy array of shape (len(x0), len(t_list)) containing the
        solutions.
    '''
    x_sol = np.zeros((len(x0), len(t_list)))
    x_sol[:, 0] = x0

    for i in range(1, len(t_list)):
        dti = t_list[i] - t_list[i - 1]
        fi = f(t_list[i - 1], x_sol[:, i - 1])
        x_sol[:, i] = x_sol[:, i - 1] + dti * fi

    return x_sol
