% This example script is supposed to show the usage of the parameter
% estimator, based on modulating functions, for a continuous system, that
% is described by an ODE of second order (linear oscillator). 
%
% m q''(t) + d q'(t) + c q(t) = u(t)  with  m,d,c > 0  and  u(t)=sin(wt)
%
% It is simple to demonstrate the idea, rather than reproducing all results
% from the paper. Further extension, such as resonance tracking, noise,
% etc. can by added once the basic idea is understood.
%
% Note that
%   - this example is only for the demonstration of modulation functions,
%     for linear systems one typically prefers, identifying the
%     coefficients of the difference equation (sampling) and then
%     converting them to the physical parameters,
%     see e.g. [Lunze: Regelungstechnik, Band II, Kapitel 4], 
%     however for nonlinear systems the mod. functions seem promising.
%   - in steady state there is trouble identifying three parameters,
%     using only one excitation frequency
%
%
% Supplementary material for the article
% "A Resonance Tracking Approach from Adaptive Control"
% Meike Gierig (Uni Hannover), Lars Flessing (TU Chemnitz)
% GAMM Archive for Students (GAMMAS), 1(1):6-17, 2019
% DOI: 10.14464/gammas.v1i1.344 
clear; close all; clc;

%% Overall Setup 

% the parameters for the system that are to be determined
m = 2;  % mass
d = 1;  % dash-pot
c = 10; % spring

omega = sqrt(c/m);  % undamped radian eigenfrequency
T=2*pi/omega;   %  period of free oscillation

te = 3.5*T;     % end time (integer multiples of T cause trouble in general
                % ..and too long measurement times (steady state) 
                % ..cause trouble for estimating 3 parameters simultanously
h = (1/100)*T;  % time step
time = 0:h:te;

u = @(tt) sin(omega*tt);  % excitation function, ideally at resonance

x0 = [0; 0];    % initial condition,   x=[position; velocity]

sys = @(tt,xx) [xx(2); u(tt)/m-(c/m)*xx(1)-(d/m)*xx(2)]; % system ode

% calculate the system response and select q(t) as measurement
[t, x] = ode45(sys, time, x0);

% the simulated "measurement" 
q = x(:,1);  % extract the positions
u_m = u(t);     % excitation values at time grid


%% Algorithmic Setup
% define the weighting functions (those are three cos-based functions to
% easily fulfill the properties (w=w'=0)

% w1: period is the same as measurement time
def_w_1 = @(tt) cos(((2*pi)/te)*tt)-1;
def_dw_dt_1 = @(tt) -((2*pi)/te)*sin(((2*pi)/te)*tt);
def_ddw_dtt_1 = @(tt) -(((2*pi)/te)^2)*cos(((2*pi)/te)*tt);

% w2: period is half the measurement time
def_w_2 = @(tt) cos(((4*pi)/te)*tt)-1;
def_dw_dt_2 = @(tt) -((4*pi)/te)*sin(((4*pi)/te)*tt);
def_ddw_dtt_2 = @(tt) -(((4*pi)/te)^2)*cos(((4*pi)/te)*tt);

% w3: period is one third of the measurement time
def_w_3 = @(tt) cos(((6*pi)/te)*tt)-1;
def_dw_dt_3 = @(tt) -((6*pi)/te)*sin(((6*pi)/te)*tt);
def_ddw_dtt_3 = @(tt) -(((6*pi)/te)^2)*cos(((6*pi)/te)*tt);

% get the actual values
w_1 = def_w_1(t);
dw_dt_1 = def_dw_dt_1(t);
ddw_dtt_1 = def_ddw_dtt_1(t);

w_2 = def_w_2(t);
dw_dt_2 = def_dw_dt_2(t);
ddw_dtt_2 = def_ddw_dtt_2(t);

w_3 = def_w_3(t);
dw_dt_3 = def_dw_dt_3(t);
ddw_dtt_3 = def_ddw_dtt_3(t);


%% Parameter Estimation Algorithm 
% setup the matrix for the equation system (lhs):
% integrate along the product of all measured values and the appropriate
% derivative of the weight functions

I_1_1 = trapz(ddw_dtt_1.*q);
I_1_2 = trapz(-dw_dt_1.*q);
I_1_3 = trapz(w_1.*q);

I_2_1 = trapz(ddw_dtt_2.*q);
I_2_2 = trapz(-dw_dt_2.*q);
I_2_3 = trapz(w_2.*q);

I_3_1 = trapz(ddw_dtt_3.*q);
I_3_2 = trapz(-dw_dt_3.*q);
I_3_3 = trapz(w_3.*q);

% coefficient matrix 
coeff = [ I_1_1, I_1_2, I_1_3;...
        I_2_1, I_2_2, I_2_3;...
        I_3_1, I_3_2, I_3_3];

% right hand side
rhs = [ trapz(w_1.*u_m);...
        trapz(w_2.*u_m);...
        trapz(w_3.*u_m)];

% solve the equation system
params = coeff \ rhs;     % backslash does automatically least squares

% compare the estimated parameters with their real values
m_d = params(1);
d_d = params(2);
c_d = params(3);


%% Post-processing
figure;
plot(t,q);  % transient oscillations are good for estimating 3 parameters
xlabel('t'); ylabel('q');

figure;
maxval=max([m,c,d, m_d,c_d,d_d]);   % for ideal line
plot([0, maxval],[0, maxval],'k', [m,d,c], [m_d,d_d,c_d],'ko'); % point should be on line with slope 1
xlabel('real mdc'); ylabel('estimated mdc'); axis equal;

% for further processing of plots in LaTeX with we recommend matlab2tikz,
% if it is installed (not part of matlab toolboxes), then use
%matlab2tikz('fig_mdc.tex', 'height', '\figureheight', 'width', '\figurewidth' );
% possibly with   cleanfigure;   before to obtain a graphic, which
% can be included in LaTeX documents (requiring tikz- and pgfplots package) 
%\setlength\figurewidth{0.8\linewidth} 
%\setlength\figureheight{0.4\linewidth} 
%\input{fig_mdc.tex}

if m>0
    e_m = abs(m_d-m)/m; 
    disp(['relative error of estimated parameters:',...
          '   e_m=',num2str(e_m)]);
else
    e_m = abs(m_d-m);
    disp(['absolute error of estimated parameters:',...
          '   e_m=',num2str(e_m),'  (m=',num2str(m),')']);
end    

if d>0  
    e_d = abs(d_d-d)/d; 
    disp(['relative error of estimated parameters:',...
          '   e_d=',num2str(e_d)]);
else
    e_d = abs(d_d-d);
        disp(['absolute error of estimated parameters:',...
          '   e_d=',num2str(e_d),'  (d=',num2str(d),')']);
end

if c>0    
    e_c = abs(c_d-c)/c;
    disp(['relative error of estimated parameters:',...
          '   e_c=',num2str(e_c)]);
else
    e_c = abs(c_d-c);
    disp(['absolute error of estimated parameters:',...
          '   e_c=',num2str(e_c),'  (c=',num2str(c),')']);
end
